#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <grp.h>
#include <dirent.h>
#include <string.h>
#include <sys/stat.h>
#include <errno.h>

static int verbose = 0;

static void checked_chown(const char *path, uid_t uid, gid_t gid)
{
	if (chown(path, uid, gid) != 0)
		fprintf(stderr, "chown %s %d %d failed with error %d\n",
			path, uid, gid, errno);
	else if (verbose)
		printf("chown %s %d %d\n", path, uid, gid);
}

static void checked_chmod(const char *path, mode_t perms)
{
	if (chmod(path, perms) != 0)
		fprintf(stderr, "chmod %s 0%3o failed with error %d\n",
			path, perms, errno);
	else if (verbose)
		printf("chmod %s 0%3o\n", path, perms);
}

#define MTD_PDDUSER_DIR "/dev/mtd3"
#define MTD_PDDKERNEL_DIR "/dev/mtd4"
#define MTD_PERMS 0660
#define MTD_GROUP "eco_pdd"

/* Iterate over all device nodes and assign mtd3 and mtd4 to user root and
   group eco_pdd, also set access rights to user and group RW. */
static void do_mtd(void)
{
	struct group *grp;
	uid_t uid_root = 0;
	gid_t gid_mtd;
	grp = getgrnam(MTD_GROUP);
	if (grp == NULL ) {
		fprintf(stderr, "no group named \'%s\' found\n", MTD_GROUP);
		return;
	}
	gid_mtd = grp->gr_gid;

	if (access(MTD_PDDUSER_DIR, F_OK) == 0) {
		checked_chown(MTD_PDDUSER_DIR, uid_root, gid_mtd);
		checked_chmod(MTD_PDDUSER_DIR, MTD_PERMS);
	}

	if (access(MTD_PDDKERNEL_DIR, F_OK) == 0) {
		checked_chown(MTD_PDDKERNEL_DIR, uid_root, gid_mtd);
		checked_chmod(MTD_PDDKERNEL_DIR, MTD_PERMS);
	}
}

#define MAX_PATH 4096

#define GPIO_DIR "/sys/class/gpio"
#define GPIO_PERMS 0660
#define GPIO_GROUP "gpio"
#define GPIO_PREFIX "gpio"

/* Iterate over all exported GPIOS and assign them to user root and
   group gpio, also set access rights to user and group RW. */
static void do_gpio(void)
{
	struct group *grp;
	uid_t uid_root = 0;
	gid_t gid_gpio;
	DIR *dir;
	struct dirent *d;
	char path[MAX_PATH];

	grp = getgrnam(GPIO_GROUP);
	if (grp == NULL ) {
		fprintf(stderr, "no group named \'%s\' found\n", GPIO_GROUP);
		return;
	}
	gid_gpio = grp->gr_gid;

	dir = opendir(GPIO_DIR);
	if (dir == NULL) {
		fprintf(stderr, "unable to open directory %s\n", GPIO_DIR);
		return;
	}

	while ((d = readdir(dir)) != NULL) {
		if (strncmp(d->d_name, GPIO_PREFIX, strlen(GPIO_PREFIX)) != 0)
			continue;

		snprintf(path, MAX_PATH, "%s/%s/value",
			 GPIO_DIR, d->d_name);
		if (access(path, F_OK) == 0) {
			checked_chown(path, uid_root, gid_gpio);
			checked_chmod(path, GPIO_PERMS);
		}

		snprintf(path, MAX_PATH, "%s/%s/edge",
			 GPIO_DIR, d->d_name);
		if (access(path, F_OK) == 0) {
			checked_chown(path, uid_root, gid_gpio);
			checked_chmod(path, GPIO_PERMS);
		}
	}

	closedir(dir);
}

#define PWM_DIR "/sys/class/pwm/pwmchip4/pwm0"
#define PWM_PERMS 0660
#define PWM_GROUP "eco_pwm"

static void do_pwm(void)
{
	struct group *grp;
	uid_t uid_root = 0;
	gid_t gid_eco_pwm;
	char path[MAX_PATH];

	grp = getgrnam(PWM_GROUP);
	if (grp == NULL) {
		fprintf(stderr, "no group named \'%s\' found\n", PWM_GROUP);
		return;
	}
	gid_eco_pwm = grp->gr_gid;

	snprintf(path, MAX_PATH, "%s/period", PWM_DIR);
	if (access(path, F_OK) == 0) {
		checked_chown(path, uid_root, gid_eco_pwm);
		checked_chmod(path, PWM_PERMS);
	}

	snprintf(path, MAX_PATH, "%s/duty_cycle", PWM_DIR);
	if (access(path, F_OK) == 0) {
		checked_chown(path, uid_root, gid_eco_pwm);
		checked_chmod(path, PWM_PERMS);
	}
}

/* Assign the t25 attribute of i2c device 0-004a (Atmel touch screen)
 * to the group i2c */
#define I2C_PATH "/sys/devices/soc0/soc.0/" \
		 "2100000.aips-bus/21a0000.i2c/i2c-0/0-004a/t25"

/* Assign the t25 attribute of i2c device 0-0064 (Atmel touch screen)
 * to the group i2c */
#define I2C_PATH_1 "/sys/devices/soc0/soc.0/" \
		 "2100000.aips-bus/21a0000.i2c/i2c-0/0-0064/t25"

/* Assign the t25 attribute of i2c device 1-0064 (Atmel touch screen)
 * to the group i2c */
#define I2C_PATH_2 "/sys/devices/soc0/soc.0/" \
		 "2100000.aips-bus/21a4000.i2c/i2c-1/1-0064/t25"

/* Assign the reset attribute of i2c device 1-0050 (Neonode touch screen)
 * to the group i2c */
#define I2C_PATH_3 "/sys/devices/soc0/soc.0/" \
		 "2100000.aips-bus/21a4000.i2c/i2c-1/1-0050/input/input4/" \
		 "zforce_reset"

/* Assign the t38_data attribute of i2c device 0-0064 (Atmel touch screen)
 * to the group i2c */
#define I2C_PATH_4 "/sys/devices/soc0/soc.0/" \
		 "2100000.aips-bus/21a0000.i2c/i2c-0/0-0064/t38_data"

/* Assign the t38_data attribute of i2c device 1-0064 (Atmel touch screen)
 * to the group i2c */
#define I2C_PATH_5 "/sys/devices/soc0/soc.0/" \
		 "2100000.aips-bus/21a4000.i2c/i2c-1/1-0064/t38_data"

#define I2C_PERMS 0660
#define I2C_GROUP "i2c"


static void do_i2c(void)
{
	struct group *grp;
	uid_t uid_root = 0;
	gid_t gid_i2c;

	grp = getgrnam(I2C_GROUP);
	if (grp == NULL) {
		fprintf(stderr, "no group named \'%s\' found\n", I2C_GROUP);
		return;
	}
	gid_i2c = grp->gr_gid;

	if (access(I2C_PATH, F_OK) == 0) {
		checked_chown(I2C_PATH, uid_root, gid_i2c);
		checked_chmod(I2C_PATH, I2C_PERMS);
	}

	if (access(I2C_PATH_1, F_OK) == 0) {
		checked_chown(I2C_PATH_1, uid_root, gid_i2c);
		checked_chmod(I2C_PATH_1, I2C_PERMS);
	}

	if (access(I2C_PATH_2, F_OK) == 0) {
		checked_chown(I2C_PATH_2, uid_root, gid_i2c);
		checked_chmod(I2C_PATH_2, I2C_PERMS);
	}

	if (access(I2C_PATH_3, F_OK) == 0) {
		checked_chown(I2C_PATH_3, uid_root, gid_i2c);
		checked_chmod(I2C_PATH_3, I2C_PERMS);
	}

	if (access(I2C_PATH_4, F_OK) == 0) {
		checked_chown(I2C_PATH_4, uid_root, gid_i2c);
		checked_chmod(I2C_PATH_4, I2C_PERMS);
	}

	if (access(I2C_PATH_5, F_OK) == 0) {
		checked_chown(I2C_PATH_5, uid_root, gid_i2c);
		checked_chmod(I2C_PATH_5, I2C_PERMS);
	}
}

#define UDROP_DIR "/sys/devices/soc0"
#define UDROP_PREFIX "undervoltage."
#define UDROP_GROUP "eco_undervoltage"

/* Find the undervoltage device under /sys and assign the attributes
 * quiescent_state and trigger to the group eco_undervoltage. */
static void do_undervoltage(void)
{
	struct group *grp;
	uid_t uid_root = 0;
	gid_t gid_udrop;
	DIR *dir;
	struct dirent *d;
	char path[MAX_PATH];

	grp = getgrnam(UDROP_GROUP);
	if (grp == NULL ) {
		fprintf(stderr, "no group named \'%s\' found\n",
			UDROP_GROUP);
		return;
	}
	gid_udrop = grp->gr_gid;

	dir = opendir(UDROP_DIR);
	if (dir == NULL) {
		fprintf(stderr, "unable to open directory %s\n",
			UDROP_DIR);
		return;
	}

	while ((d = readdir(dir)) != NULL) {
		if (strncmp(d->d_name, UDROP_PREFIX, strlen(UDROP_PREFIX)) != 0)
			continue;

		snprintf(path, MAX_PATH, "%s/%s/quiescent_state",
			 UDROP_DIR, d->d_name);
		if (access(path, F_OK) == 0) {
			checked_chown(path, uid_root, gid_udrop);
		}

		snprintf(path, MAX_PATH, "%s/%s/trigger",
			 UDROP_DIR, d->d_name);
		if (access(path, F_OK) == 0) {
			checked_chown(path, uid_root, gid_udrop);
		}
	}

	closedir(dir);
}

/* Helper function to find all character devices in directory DIR_NAME
 * and assign the to group GROUP_NAME and change their permissions to
 * PERMS. */
static void do_char_devices(char *dir_name, char *group_name, int perms)
{
       	struct group *grp;
	uid_t uid_root = 0;
	gid_t gid;
	DIR *dir;
	struct dirent *d;
	char path[MAX_PATH];
	struct stat st;

	grp = getgrnam(group_name);
	if (grp == NULL ) {
		fprintf(stderr, "no group named \'%s\' found\n",
			group_name);
		return;
	}
	gid = grp->gr_gid;

	dir = opendir(dir_name);
	if (dir == NULL) {
		fprintf(stderr, "unable to open directory %s\n",
			dir_name);
		return;
	}

	while ((d = readdir(dir)) != NULL) {
		snprintf(path, MAX_PATH, "%s/%s", dir_name, d->d_name);
		if (stat(path, &st) != 0)
			continue;
		if (!S_ISCHR(st.st_mode))
			continue;

		checked_chown(path, uid_root, gid);
		checked_chmod(path, perms);
	}

	closedir(dir);
}

#define ADC_DIR "/dev/adc"
#define ADC_PERMS 0660
#define ADC_GROUP "adc"

/* Assign all charcter devices in /dev/adc to the group adc and set
 * their group access rights to read/write. */
static void do_adc(void)
{
	do_char_devices(ADC_DIR, ADC_GROUP, ADC_PERMS);
}

#define INPUT_DIR "/dev/input"
#define INPUT_PERMS 0660
#define INPUT_GROUP "input"

/* Assign all charcter devices in /dev/input to the group input and set
 * their group access rights to read/write. */
static void do_input(void)
{
	do_char_devices(INPUT_DIR, INPUT_GROUP, INPUT_PERMS);
}

#define DEBUG_PATH_1  "/sys/kernel/debug/gc/meminfo"
#define DEBUG_PATH_2  "/sys/kernel/debug/gc/idle"
#define DEBUG_PATH_3  "/sys/kernel/debug/gc/database"
#define DEBUG_PATH_4  "/sys/kernel/debug/gc/clients"
#define DEBUG_PATH_5  "/sys/kernel/debug/gc/vidmem"
#define DEBUG_PATH_6  "/proc/kmsg"
#define DEBUG_PATH_7  "/sys/kernel/debug"
#define DEBUG_PATH_8  "/sys/kernel/debug/gc"

#define PROC_R_PERMS 	0440
#define DEBUG_R_PERMS 	0640
#define DEBUG_RW_PERMS 	0660
#define DEBUG_RX_PERMS 	0750
#define DEBUG_GROUP 	"eco_debug"

/* Assign the /sys/kernel/debug/gc/ path permissions to the group
 * eco_debug */
static void do_debug(void)
{
	struct group *grp;
	uid_t uid_root = 0;
	gid_t gid_debug;

	grp = getgrnam(DEBUG_GROUP);
	if (grp == NULL)
	{
		fprintf(stderr, "no group named \'%s\' found\n", DEBUG_GROUP);
		return;
	}
	gid_debug = grp->gr_gid;

	if (access(DEBUG_PATH_1, F_OK) == 0)
	{
		checked_chown(DEBUG_PATH_1, uid_root, gid_debug);
		checked_chmod(DEBUG_PATH_1, DEBUG_R_PERMS);
	}

	if (access(DEBUG_PATH_2, F_OK) == 0)
	{
		checked_chown(DEBUG_PATH_2, uid_root, gid_debug);
		checked_chmod(DEBUG_PATH_2, DEBUG_R_PERMS);
	}

	if (access(DEBUG_PATH_3, F_OK) == 0)
	{
		checked_chown(DEBUG_PATH_3, uid_root, gid_debug);
		checked_chmod(DEBUG_PATH_3, DEBUG_R_PERMS);
	}

	if (access(DEBUG_PATH_4, F_OK) == 0)
	{
		checked_chown(DEBUG_PATH_4, uid_root, gid_debug);
		checked_chmod(DEBUG_PATH_4, DEBUG_R_PERMS);
	}

	if (access(DEBUG_PATH_5, F_OK) == 0)
	{
		checked_chown(DEBUG_PATH_5, uid_root, gid_debug);
		checked_chmod(DEBUG_PATH_5, DEBUG_RW_PERMS);
	}

	if (access(DEBUG_PATH_6, F_OK) == 0)
	{
		checked_chown(DEBUG_PATH_6, uid_root, gid_debug);
		checked_chmod(DEBUG_PATH_6, PROC_R_PERMS);
	}
	if (access(DEBUG_PATH_7, F_OK) == 0)
	{
		checked_chown(DEBUG_PATH_7, uid_root, gid_debug);
		checked_chmod(DEBUG_PATH_7, DEBUG_RX_PERMS);
	}
	if (access(DEBUG_PATH_8, F_OK) == 0)
	{
		checked_chown(DEBUG_PATH_8, uid_root, gid_debug);
		checked_chmod(DEBUG_PATH_8, DEBUG_RX_PERMS);
	}
}

static void do_help(void)
{
	printf("modify_gpio_adc_input_perms\n");
	printf("\t-v\tverbose mode\n");
	printf("\t-h\tdisplay help text\n");
}

int main(int argc, char **argv)
{
	int opt;

	while ((opt = getopt(argc, argv, "vh")) != -1) {
		switch (opt) {
		case 'v':
			verbose = 1;
			printf("verbose mode\n");
			break;
		case 'h':
			do_help();
			exit(0);
		}
	}

	do_i2c();
	do_gpio();
	do_undervoltage();
	do_pwm();
	do_adc();
	do_input();
	do_mtd();
	do_debug();

	return 0;
}
